// ====================== FOREX CALCULATOR CORE ======================

// Helper shortcuts
const $ = (id) => document.getElementById(id);
const n = (v, fb = 0) => {
  const x = parseFloat(v);
  return Number.isFinite(x) ? x : fb;
};
const fmt = (v, d = 2) =>
  Number.isFinite(v)
    ? v.toLocaleString(undefined, {
        minimumFractionDigits: d,
        maximumFractionDigits: d,
      })
    : "—";

// Globals
let lotUnit = "standard";
let lastRiskChanged = "percent";
let category = "metals";
let symbol = "XAUUSD";

let POINT_SIZE = 0.01;
let PIP_SIZE = 0.10;
let PIP_VALUE_STD = 10;
let CONTRACT_STD_UNITS = 100;
let PRICE_PRECISION = 2;

// Lot and leverage
const LOT_SCALE = { standard: 1, mini: 0.1, micro: 0.01 };
const MIN_LOT = { standard: 0.01, mini: 0.1, micro: 0.001 };

// Leverage helper
function leverageValue() {
  const txt = $("leverage").value;
  return n(txt.split(":")[1], 100);
}

// Default specs
const FX_5_DIGIT = { point: 0.00001, pip: 0.0001, pipValueStd: 10, contractStdUnits: 100000, precision: 5 };
const FX_3_DIGIT_JPY = { point: 0.001, pip: 0.01, pipValueStd: 10, contractStdUnits: 100000, precision: 3 };

// Catalog of instruments
const CATALOG = {
  metals: {
    name: "Metals",
    symbols: {
      XAUUSD: { point: 0.01, pip: 0.10, pipValueStd: 10, contractStdUnits: 100, precision: 2 },
      XAGUSD: { point: 0.01, pip: 0.10, pipValueStd: 5, contractStdUnits: 5000, precision: 3 },
      XPTUSD: { point: 0.01, pip: 0.10, pipValueStd: 10, contractStdUnits: 100, precision: 2 },
      XPDUSD: { point: 0.01, pip: 0.10, pipValueStd: 10, contractStdUnits: 100, precision: 2 },
    },
  },
  majors: {
    name: "Majors",
    symbols: {
      EURUSD: FX_5_DIGIT, GBPUSD: FX_5_DIGIT, AUDUSD: FX_5_DIGIT, NZDUSD: FX_5_DIGIT,
      USDJPY: FX_3_DIGIT_JPY, USDCHF: FX_5_DIGIT, USDCAD: FX_5_DIGIT,
    },
  },
  minors: {
    name: "Minors",
    symbols: {
      EURGBP: FX_5_DIGIT, EURJPY: FX_3_DIGIT_JPY, EURCHF: FX_5_DIGIT, EURAUD: FX_5_DIGIT, EURNZD: FX_5_DIGIT, EURCAD: FX_5_DIGIT,
      GBPJPY: FX_3_DIGIT_JPY, GBPCHF: FX_5_DIGIT, GBPAUD: FX_5_DIGIT, GBPCAD: FX_5_DIGIT, GBPNZD: FX_5_DIGIT,
      AUDJPY: FX_3_DIGIT_JPY, AUDNZD: FX_5_DIGIT, AUDCAD: FX_5_DIGIT, AUDCHF: FX_5_DIGIT,
      NZDJPY: FX_3_DIGIT_JPY, NZDCAD: FX_5_DIGIT, NZDCHF: FX_5_DIGIT,
      CADJPY: FX_3_DIGIT_JPY, CADCHF: FX_5_DIGIT, CHFJPY: FX_3_DIGIT_JPY,
    },
  },
  exotics: {
    name: "Exotics",
    symbols: {
      USDZAR: { point: 0.0001, pip: 0.001, pipValueStd: 1, contractStdUnits: 100000, precision: 4 },
      USDTRY: { point: 0.0001, pip: 0.001, pipValueStd: 1, contractStdUnits: 100000, precision: 4 },
      USDSEK: { point: 0.0001, pip: 0.001, pipValueStd: 1, contractStdUnits: 100000, precision: 4 },
      USDNOK: { point: 0.0001, pip: 0.001, pipValueStd: 1, contractStdUnits: 100000, precision: 4 },
      USDMXN: { point: 0.0001, pip: 0.001, pipValueStd: 1, contractStdUnits: 100000, precision: 4 },
      USDPLN: { point: 0.0001, pip: 0.001, pipValueStd: 1, contractStdUnits: 100000, precision: 4 },
      USDHUF: { point: 0.01, pip: 0.1, pipValueStd: 1, contractStdUnits: 100000, precision: 2 },
      USDTHB: { point: 0.0001, pip: 0.001, pipValueStd: 1, contractStdUnits: 100000, precision: 4 },
      USDHKD: { point: 0.00001, pip: 0.0001, pipValueStd: 10, contractStdUnits: 100000, precision: 5 },
      USDINR: { point: 0.00001, pip: 0.0001, pipValueStd: 10, contractStdUnits: 100000, precision: 5 },
    },
  },
  commodities: {
    name: "Commodities",
    symbols: {
      XTIUSD: { point: 0.01, pip: 0.10, pipValueStd: 10, contractStdUnits: 1000, precision: 2 },
      XBRUSD: { point: 0.01, pip: 0.10, pipValueStd: 10, contractStdUnits: 1000, precision: 2 },
    },
  },
};

// ============ TOAST NOTIFICATION SYSTEM ============
function showToast(message, type = 'info') {
  const container = $('toast-container');
  const toast = document.createElement('div');
  toast.className = `toast toast-${type}`;
  
  const icons = {
    success: '<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="20 6 9 17 4 12"/></svg>',
    error: '<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"/><line x1="15" y1="9" x2="9" y2="15"/><line x1="9" y1="9" x2="15" y2="15"/></svg>',
    info: '<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"/><path d="M12 16v-4"/><path d="M12 8h.01"/></svg>',
    warning: '<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"/><line x1="12" y1="9" x2="12" y2="13"/><line x1="12" y1="17" x2="12.01" y2="17"/></svg>'
  };
  
  toast.innerHTML = `
    ${icons[type]}
    <span>${message}</span>
  `;
  
  container.appendChild(toast);
  
  setTimeout(() => toast.classList.add('show'), 10);
  
  setTimeout(() => {
    toast.classList.remove('show');
    setTimeout(() => toast.remove(), 300);
  }, 3000);
}

// ============ PWA INSTALL PROMPT ============
let deferredPrompt;

window.addEventListener('beforeinstallprompt', (e) => {
  e.preventDefault();
  deferredPrompt = e;
  
  const prompt = $('install-prompt');
  if (prompt) {
    prompt.classList.remove('hidden');
    setTimeout(() => prompt.classList.add('show'), 100);
  }
});

if ($('install-btn')) {
  $('install-btn').addEventListener('click', async () => {
    if (!deferredPrompt) return;
    
    deferredPrompt.prompt();
    const { outcome } = await deferredPrompt.userChoice;
    
    if (outcome === 'accepted') {
      showToast('App installed successfully!', 'success');
    }
    
    deferredPrompt = null;
    $('install-prompt').classList.remove('show');
    setTimeout(() => $('install-prompt').classList.add('hidden'), 300);
  });
}

if ($('close-install')) {
  $('close-install').addEventListener('click', () => {
    $('install-prompt').classList.remove('show');
    setTimeout(() => $('install-prompt').classList.add('hidden'), 300);
  });
}

// ============ INSTRUMENT MANAGEMENT ============
function populateSymbols(catKey) {
  const sel = $("symbol");
  sel.innerHTML = "";
  Object.keys(CATALOG[catKey].symbols).forEach((s) => {
    const o = document.createElement("option");
    o.value = s;
    o.textContent = s;
    sel.appendChild(o);
  });
}

function applyInstrument(catKey, symKey) {
  category = catKey;
  symbol = symKey;
  const spec = CATALOG[catKey].symbols[symKey];
  
  POINT_SIZE = spec.point || FX_5_DIGIT.point;
  PIP_SIZE = spec.pip || FX_5_DIGIT.pip;
  PIP_VALUE_STD = spec.pipValueStd || FX_5_DIGIT.pipValueStd;
  CONTRACT_STD_UNITS = spec.contractStdUnits || FX_5_DIGIT.contractStdUnits;
  PRICE_PRECISION = spec.precision || FX_5_DIGIT.precision;
  
  const std = CONTRACT_STD_UNITS.toLocaleString();
  const mini = (CONTRACT_STD_UNITS * 0.1).toLocaleString();
  const micro = (CONTRACT_STD_UNITS * 0.01).toLocaleString();
  $("lot-hint").textContent = `Standard ${std} · Mini ${mini} · Micro ${micro} units`;

  updateUI();
  showToast(`Switched to ${symKey} (${CATALOG[catKey].name})`, 'info');
}

function inferDirection(entry, sl, tp) {
  if (!Number.isFinite(entry) || !Number.isFinite(sl) || !Number.isFinite(tp))
    return "UNKNOWN";
  if (tp === entry || sl === entry || sl === tp) return "INVALID";
  if (tp > entry && sl < entry) return "BUY";
  if (tp < entry && sl > entry) return "SELL";
  return "INVALID";
}

// ============ CORE UPDATE FUNCTION ============
function updateUI() {
  const balance = n($("balance").value, 0);
  const lots = Math.max(0, n($("lots").value, 0));
  const entry = n($("entry").value);
  const sl = n($("sl").value);
  const tp = n($("tp").value);
  const lev = leverageValue();
  const pricePrecision = PRICE_PRECISION;

  // Risk sync
  let riskAmount = Math.max(0, n($("risk-amount").value, 0));
  let riskPercent = Math.max(0, n($("risk-percent").value, 0));
  if (lastRiskChanged === "percent") {
    riskAmount = balance * (riskPercent / 100);
    $("risk-amount").value = fmt(riskAmount, 2);
  } else {
    riskPercent = balance > 0 ? (riskAmount / balance) * 100 : 0;
    $("risk-percent").value = fmt(riskPercent, 2);
  }

  // Direction & Distances
  const slDistance = Math.abs(entry - sl);
  const tpDistance = Math.abs(tp - entry);

  const dir = inferDirection(entry, sl, tp);
  const banner = $("banner");
  const title = $("banner-title");
  const bannerDesc = $("banner-desc");
  const bannerNums = $("banner-nums");
  
  banner.className = `banner ${dir.toLowerCase()}`;
  
  const directionInfo = {
    BUY: {
      title: 'BUY SIGNAL',
      desc: 'Long position detected. You\'re anticipating a price increase. SL is below entry, TP is above.',
      icon: '<svg width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><line x1="12" y1="19" x2="12" y2="5"/><polyline points="5 12 12 5 19 12"/></svg>'
    },
    SELL: {
      title: 'SELL SIGNAL',
      desc: 'Short position detected. You\'re anticipating a price drop. SL is above entry, TP is below.',
      icon: '<svg width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><line x1="12" y1="5" x2="12" y2="19"/><polyline points="19 12 12 19 5 12"/></svg>'
    },
    INVALID: {
      title: 'INVALID SETUP',
      desc: 'Configuration error. SL and TP must be on opposite sides of the Entry price.',
      icon: '<svg width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"/><line x1="12" y1="9" x2="12" y2="13"/><line x1="12" y1="17" x2="12.01" y2="17"/></svg>'
    },
    UNKNOWN: {
      title: 'AWAITING INPUT',
      desc: 'Set Entry, SL, and TP prices to auto-detect trade direction.',
      icon: '<svg width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><circle cx="12" cy="12" r="10"/><path d="M12 16v-4"/><path d="M12 8h.01"/></svg>'
    }
  };
  
  const info = directionInfo[dir];
  title.textContent = info.title;
  bannerDesc.textContent = info.desc;
  
  // Update banner icon
  const iconElement = banner.querySelector('.b-icon');
  if (iconElement) {
    iconElement.innerHTML = info.icon;
  }

  const fmtPrice = (val) => fmt(val, pricePrecision);
  bannerNums.textContent = `Entry: ${fmtPrice(entry)} · SL: ${fmtPrice(sl)} · TP: ${fmtPrice(tp)}`;
  
  $("btn-buy").disabled = dir === "SELL" || dir === "INVALID";
  $("btn-sell").disabled = dir === "BUY" || dir === "INVALID";

  // Pip/Point Calculations
  const pipsSL = slDistance / PIP_SIZE;
  const pipsTP = tpDistance / PIP_SIZE;
  const pointsSL = slDistance / POINT_SIZE;
  const pointsTP = tpDistance / POINT_SIZE;

  $("sl-price-diff").textContent = fmtPrice(slDistance);
  $("sl-points").textContent = fmt(pointsSL, 1);
  $("sl-pips").textContent = fmt(pipsSL, 1);
  
  $("tp-price-diff").textContent = fmtPrice(tpDistance);
  $("tp-points").textContent = fmt(pointsTP, 1);
  $("tp-pips").textContent = fmt(pipsTP, 1);

  // Trade Outcome Calculations
  const pipValuePerLot = PIP_VALUE_STD * LOT_SCALE[lotUnit];
  const loss = pipsSL * pipValuePerLot * lots * -1;
  const gain = pipsTP * pipValuePerLot * lots;
  const rr = (gain && loss) ? Math.abs(gain / loss) : 0;

  // Margin Calculations
  const unitsPerLot = CONTRACT_STD_UNITS * LOT_SCALE[lotUnit];
  const notional = entry * (lots * unitsPerLot);
  const margin = lev > 0 ? notional / lev : 0;
  const freeMargin = balance - margin;
  const balLoss = balance + loss;
  const balWin = balance + gain;

  // Update Results UI
  $("loss").textContent = `$${fmt(Math.abs(loss), 2)}`;
  $("gain").textContent = `$${fmt(gain, 2)}`;
  $("rr").textContent = rr ? `${fmt(rr, 2)} : 1` : "—";
  $("margin").textContent = `$${fmt(margin, 2)}`;
  $("notional").textContent = `$${fmt(notional, 2)}`;
  $("levtxt").textContent = `1:${lev}`;
  $("free-margin").textContent = `$${fmt(freeMargin, 2)}`;
  $("bal-loss").textContent = `$${fmt(balLoss, 2)}`;
  $("bal-win").textContent = `$${fmt(balWin, 2)}`;
  $("risk-target").textContent = `$${fmt(riskAmount, 2)}`;

  // Suggested lots
  const denom = pipsSL * pipValuePerLot;
  const suggested = denom > 0 ? riskAmount / denom : 0;
  
  $("risk-amt-inline").textContent = `$${fmt(riskAmount, 2)}`;
  $("lots-suggest").textContent = fmt(suggested, 2);
  $("apply-suggest").onclick = () => {
    $("lots").value = suggested.toFixed(2);
    showToast(`Applied suggested lot size: ${suggested.toFixed(2)}`, 'success');
    updateUI();
  };
  
  // Warning for insufficient margin
  if (margin > balance) {
    showToast('Warning: Required margin exceeds account balance!', 'warning');
  }
}

// ============ MIN LOT APPLICATION ============
function applyMinLot() {
  const min = MIN_LOT[lotUnit] || 0.01;
  const lotsEl = $("lots");
  lotsEl.min = String(min);
  lotsEl.step = String(min);
  
  const decimals = String(min).split('.')[1]?.length || 2;
  $("lot-min-hint").textContent = `Min lot: ${fmt(min, decimals)}`;

  if (n(lotsEl.value, 0) < min) {
    lotsEl.value = String(min);
  }
}

// ============ EVENT HOOKS ============
["balance", "lots", "entry", "sl", "tp"].forEach((id) =>
  $(id).addEventListener("input", updateUI)
);

$("leverage").addEventListener("change", updateUI);

$("risk-amount").addEventListener("input", () => {
  lastRiskChanged = "amount";
  updateUI();
});

$("risk-percent").addEventListener("input", () => {
  lastRiskChanged = "percent";
  updateUI();
});

document.querySelectorAll("[data-lot]").forEach((btn) =>
  btn.addEventListener("click", (e) => {
    lotUnit = e.currentTarget.getAttribute("data-lot");
    document.querySelectorAll("[data-lot]").forEach((b) => b.classList.remove("active"));
    e.currentTarget.classList.add("active");
    applyMinLot();
    updateUI();
    showToast(`Lot unit changed to ${lotUnit}`, 'info');
  })
);

// Category and symbol selection
$("category").addEventListener("change", (e) => {
  const cat = e.target.value;
  populateSymbols(cat);
  const first = Object.keys(CATALOG[cat].symbols)[0];
  applyInstrument(cat, first);
});

$("symbol").addEventListener("change", (e) => {
  applyInstrument($("category").value, e.target.value);
});

// Clear button
$("btn-clear").addEventListener("click", () => {
  $("balance").value = "10000";
  $("leverage").value = "1:100";
  $("category").value = "metals";
  populateSymbols("metals");
  
  $("lots").value = "0.10";
  $("risk-amount").value = "100";
  $("risk-percent").value = "1";
  
  $("entry").value = "4200.00";
  $("sl").value = "4085.00";
  $("tp").value = "4430.00";
  
  applyInstrument("metals", "XAUUSD");
  
  lotUnit = "standard";
  document.getElementById("lot-standard").classList.add("active");
  document.getElementById("lot-mini").classList.remove("active");
  document.getElementById("lot-micro").classList.remove("active");

  lastRiskChanged = "percent";
  window.scrollTo({ top: 0, behavior: "smooth" });
  applyMinLot();
  updateUI();
  showToast('Calculator reset to defaults', 'info');
});

// Keyboard shortcuts
document.addEventListener('keydown', (e) => {
  if (e.ctrlKey || e.metaKey) {
    if (e.key === 'r' || e.key === 'R') {
      e.preventDefault();
      $('btn-clear').click();
    }
  }
});

// ============ INITIALIZATION ============
$("category").value = "metals";
populateSymbols("metals");
applyInstrument("metals", "XAUUSD");
$("lots").value = "0.10";
$("entry").value = "4200.00";
$("sl").value = "4085.00";
$("tp").value = "4430.00";
applyMinLot();
updateUI();

// Add fade-in animation on load
window.addEventListener('load', () => {
  document.querySelectorAll('.fade-in').forEach((el, i) => {
    setTimeout(() => el.style.opacity = '1', i * 100);
  });
  showToast('Calculator ready! Enter your trade details.', 'success');
});
