import yfinance as yf
import pandas as pd
import pandas_ta as ta
from datetime import datetime, timedelta
import time

# --- FORENSIC HANDBOOK RULES (VERSION 3.0) ---
PAIR_CONFIG = {
    "EURUSD=X": {
        "name": "EURUSD",
        "mode": "M15",
        "direction": "SHORT", # Strict M15 rule
        "hours_gmt": [11, 12, 13],
        "days": [1, 2, 3, 4], # Tue-Fri
        "pivot_focus": "R1"
    },
    "GBPUSD=X": {
        "name": "GBPUSD",
        "mode": "M15",
        "direction": "SHORT",
        "hours_gmt": [12, 13],
        "days": [1, 3, 4], # Tue, Thu, Fri (Wed Excluded)
        "pivot_focus": "R1"
    },
    "USDJPY=X": {
        "name": "USDJPY",
        "mode": "M15",
        "direction": "BOTH",
        "hours_gmt": [13], # Precision Hour
        "days": [1, 3, 4], # Tue, Thu, Fri
        "pivot_focus": "BOTH"
    },
    "GC=F": { 
        "name": "XAUUSD",
        "mode": "H1", # H1 Only Rule
        "direction": "BOTH",
        "hours_gmt": [9, 11, 12, 13],
        "days": [2, 3], # Wed, Thu
        "pivot_focus": "BOTH"
    },
    "USDCAD=X": {
        "name": "USDCAD",
        "mode": "H1",
        "direction": "BOTH",
        "hours_gmt": [8, 11, 12, 13],
        "days": [1, 2], # Tue, Wed
        "pivot_focus": "S1_LONG_R1_SHORT"
    },
    "AUDUSD=X": {
        "name": "AUDUSD",
        "mode": "H1",
        "direction": "LONG", # S1 Long Bias
        "hours_gmt": [2, 6, 8],
        "days": [1, 4], # Tue, Fri
        "pivot_focus": "S1"
    }
}

def get_data(ticker, interval="15m", period="5d"):
    """Fetch live data and calculate indicators"""
    try:
        df = yf.download(ticker, interval=interval, period=period, progress=False)
        if df.empty: return None

        # Handle MultiIndex columns if present
        if isinstance(df.columns, pd.MultiIndex):
            df.columns = df.columns.get_level_values(0)

        # 1. Heikin Ashi Calculation
        df['HA_Close'] = (df['Open'] + df['High'] + df['Low'] + df['Close']) / 4
        ha_open = [df['Open'].iloc[0]]
        for i in range(1, len(df)):
            ha_open.append((ha_open[i-1] + df['HA_Close'].iloc[i-1]) / 2)
        df['HA_Open'] = ha_open
        df['HA_High'] = df[['High', 'HA_Open', 'HA_Close']].max(axis=1)
        df['HA_Low'] = df[['Low', 'HA_Open', 'HA_Close']].min(axis=1)

        # 2. Indicators
        df['EMA15'] = ta.ema(df['Close'], length=15)
        
        macd = ta.macd(df['Close'])
        df['MACD'] = macd['MACD_12_26_9']
        df['Signal'] = macd['MACDs_12_26_9']
        
        stoch = ta.stoch(df['High'], df['Low'], df['Close'], k=5, d=3, smooth_k=3)
        df['Stoch_K'] = stoch['STOCHk_5_3_3']
        df['Stoch_D'] = stoch['STOCHd_5_3_3']

        return df
    except Exception as e:
        print(f"Error fetching {ticker}: {e}")
        return None

def calculate_pivots(ticker):
    """Calculate Standard Daily Pivots from yesterday's D1 data"""
    try:
        df_daily = yf.download(ticker, interval="1d", period="5d", progress=False)
        if isinstance(df_daily.columns, pd.MultiIndex):
            df_daily.columns = df_daily.columns.get_level_values(0)
        
        # Ensure we get the last COMPLETED day (iloc[-2] usually, as -1 is current incomplete day)
        prev_day = df_daily.iloc[-2]
        
        high = prev_day['High']
        low = prev_day['Low']
        close = prev_day['Close']
        
        pp = (high + low + close) / 3
        r1 = (2 * pp) - low
        s1 = (2 * pp) - high
        
        return {"PP": pp, "R1": r1, "S1": s1}
    except Exception:
        return {"PP": 0, "R1": 0, "S1": 0}

def check_sniper_conditions():
    """Run the Forensic Analysis"""
    results = []
    # Note: Server time might need adjustment to match GMT if server is not in GMT
    current_time = datetime.utcnow()
    current_hour = current_time.hour
    current_weekday = current_time.weekday() # 0=Mon, 6=Sun
    
    for ticker, config in PAIR_CONFIG.items():
        status = {
            "pair": config['name'],
            "signal": "WAIT",
            "reason": "Scanning...",
            "timestamp": current_time.strftime("%H:%M GMT"),
            "data": {}
        }

        # --- 1. Filter: Day & Time ---
        if current_weekday not in config['days']:
            status['reason'] = f"Day Filter (Allowed: {config['days']})"
            results.append(status)
            continue
            
        if current_hour not in config['hours_gmt']:
            status['reason'] = f"Time Filter (Allowed: {config['hours_gmt']} GMT)"
            results.append(status)
            continue

        # --- 2. Filter: Data Fetch ---
        interval = "1h" if config['mode'] == "H1" else "15m"
        df = get_data(ticker, interval=interval)
        pivots = calculate_pivots(ticker)
        
        if df is None:
            status['reason'] = "Data Error"
            results.append(status)
            continue

        row = df.iloc[-1]
        
        # --- 3. Filter: Technicals ---
        
        # Heikin Ashi Logic
        ha_green = row['HA_Close'] > row['HA_Open']
        # Flat bottom: Open equals Low (with tiny buffer for precision errors)
        flat_bottom = ha_green and (abs(row['HA_Open'] - row['HA_Low']) <= (row['Close'] * 0.0001))
        # Flat top: Open equals High
        flat_top = (not ha_green) and (abs(row['HA_Open'] - row['HA_High']) <= (row['Close'] * 0.0001))
        
        # Stoch Gap Logic (The 10 point rule)
        stoch_gap_long = row['Stoch_K'] > (row['Stoch_D'] + 10)
        stoch_gap_short = row['Stoch_K'] < (row['Stoch_D'] - 10)
        
        # Pivot Proximity Logic (Is price near S1 or R1?)
        # Threshold approx 5-10 pips depending on pair digit
        pip_val = 0.01 if "JPY" in ticker else 0.0001
        threshold = 10 * pip_val 
        
        at_s1 = abs(row['Low'] - pivots['S1']) < threshold or abs(row['Close'] - pivots['S1']) < threshold
        at_r1 = abs(row['High'] - pivots['R1']) < threshold or abs(row['Close'] - pivots['R1']) < threshold

        # --- EVALUATION ---
        
        # LONG CHECK
        if config['direction'] in ["LONG", "BOTH"]:
            if (flat_bottom and 
                row['Close'] > row['EMA15'] and 
                row['MACD'] > row['Signal'] and 
                stoch_gap_long and 
                at_s1):
                status['signal'] = "SNIPER BUY"
                status['reason'] = "S1 Bounce + HA Flat Bottom + Stoch Gap"

        # SHORT CHECK
        if config['direction'] in ["SHORT", "BOTH"]:
            if (flat_top and 
                row['Close'] < row['EMA15'] and 
                row['MACD'] < row['Signal'] and 
                stoch_gap_short and 
                at_r1):
                status['signal'] = "SNIPER SELL"
                status['reason'] = "R1 Rejection + HA Flat Top + Stoch Gap"
        
        # Populate Display Data
        status['data'] = {
            "Price": round(row['Close'], 4),
            "S1": round(pivots['S1'], 4),
            "R1": round(pivots['R1'], 4),
            "Stoch_K": round(row['Stoch_K'], 2),
            "Stoch_D": round(row['Stoch_D'], 2),
            "HA_Color": "Green" if ha_green else "Red"
        }
        
        results.append(status)
    
    return results